using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents remote procedure call request. Calling application should pass
	/// an initialized instance of this class to the RpcBroker to perform remote procedure call. 
	/// </summary>
	public sealed class RpcRequestMessage : VistALinkMessage
	{
		private readonly RpcRequest _request;
		private static readonly VersionNumber RpcHandlerVersion = new VersionNumber( "1.0" );

		/// <summary>
		/// VistALink message type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string MessageType = "gov.va.med.foundations.rpc.request";

		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "rpcRequest.xsd";

		// Constants used in XML message serialization/deserialization		
		private const string XMLCONSTS_RPC_HANDLER_NODE_NAME = "RpcHandler";
		private const string XMLCONSTS_VERSION_RPC_HANDLER_ATTRIBUTE_NAME = "version";

		/// <summary>
		/// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
		/// </summary>
		static RpcRequestMessage() {}

		/// <summary>
		/// Simple constructor. The only parameter required to create RPC 
		/// request message is RPC request itself. 
		/// </summary>
		/// <param name="request">RPC request to submit to remote server.</param>
		public RpcRequestMessage( RpcRequest request )
			: base( MessageType, XsdName )
		{
			if( request == null )
				throw( new ArgumentNullException( "request" ) );

			_request = request;
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		/// <param name="parameterFactory">Factory object parsing source XML and creating appropriate RPC parameter objects.</param>
		public RpcRequestMessage( XmlDocument sourceDoc, RpcParameterXmlFactory parameterFactory  )
			: base( MessageType, XsdName, sourceDoc )
		{
			// RpcHandler must be immediately before the request element
			if(	sourceDoc.DocumentElement == null || 
				sourceDoc.DocumentElement.FirstChild == null || 
				((XmlElement)sourceDoc.DocumentElement.FirstChild).Name != XMLCONSTS_RPC_HANDLER_NODE_NAME ) 
				throw( new XmlParseException( SR.Exceptions.XmlParseRequiredElementNotFound( XMLCONSTS_RPC_HANDLER_NODE_NAME )) );

			XmlElement _rpcHandlerElement = (XmlElement)sourceDoc.DocumentElement.FirstChild; 
			
			// Rpc handler version is optional, and it must be higher or equal to supported version.
			if( _rpcHandlerElement.HasAttribute( XMLCONSTS_VERSION_RPC_HANDLER_ATTRIBUTE_NAME ) && 
				RpcHandlerVersion.CompareTo( VersionNumber.Parse( _rpcHandlerElement.GetAttribute( XMLCONSTS_VERSION_RPC_HANDLER_ATTRIBUTE_NAME ) ) ) > 0 )
				throw( new XmlParseException( SR.Exceptions.RpcHandlerVersionMismatch( 
					_rpcHandlerElement.GetAttribute( XMLCONSTS_VERSION_RPC_HANDLER_ATTRIBUTE_NAME ), 
					RpcHandlerVersion.ToString() ) ) );

			_request = new RpcRequest( sourceDoc.DocumentElement, parameterFactory );
		}

		/// <summary>
		/// Standard VistALink message XML serialization method, 
		/// translating message object into XML representation.
		/// </summary>
		/// <param name="writer">XmlWriter object to use</param>
		protected override void WriteXmlMessageBody( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_RPC_HANDLER_NODE_NAME );
			writer.WriteAttributeString( XMLCONSTS_VERSION_RPC_HANDLER_ATTRIBUTE_NAME, RpcHandlerVersion.ToString() );
			writer.WriteEndElement();
			
			_request.WriteRpcRequestToXml( writer );
		}

		/// <summary>
		/// RPC request.
		/// </summary>
		public RpcRequest RpcRequest
		{
			get
			{
				return _request;
			}
		}
	}
}
